<?php
// functions.php - core logic for uptime panel

$config = require __DIR__ . '/config.php';

/**
 * ---------------------------
 * Generic storage helpers
 * ---------------------------
 */
function load_monitors(): array
{
    global $config;
    if (!file_exists($config['storage_file'])) {
        file_put_contents($config['storage_file'], json_encode([]));
    }
    $json = file_get_contents($config['storage_file']);
    $data = json_decode($json, true);
    return is_array($data) ? $data : [];
}

function save_monitors(array $monitors): void
{
    global $config;
    file_put_contents($config['storage_file'], json_encode($monitors, JSON_PRETTY_PRINT));
}

function generate_id(): string
{
    return bin2hex(random_bytes(8));
}

function now(): int
{
    return time();
}

function find_monitor(array $monitors, string $id): ?array
{
    foreach ($monitors as $m) {
        if ($m['id'] === $id) {
            return $m;
        }
    }
    return null;
}

function update_monitor_in_list(array &$monitors, array $monitor): void
{
    foreach ($monitors as &$m) {
        if ($m['id'] === $monitor['id']) {
            $m = $monitor;
            return;
        }
    }
    $monitors[] = $monitor;
}

/**
 * ---------------------------
 * Proxy helpers (B2 + B3)
 * ---------------------------
 */

function load_dynamic_proxies(): array
{
    global $config;
    $file = $config['proxies_file'];

    if (!file_exists($file)) {
        file_put_contents($file, json_encode([]));
    }

    $json = file_get_contents($file);
    $data = json_decode($json, true);
    return is_array($data) ? $data : [];
}

function save_dynamic_proxies(array $proxies): void
{
    global $config;
    $file = $config['proxies_file'];
    file_put_contents($file, json_encode($proxies, JSON_PRETTY_PRINT));
}

function get_all_proxies(): array
{
    global $config;
    $static = $config['static_proxies'] ?? [];
    $dynamic = load_dynamic_proxies();

    return array_merge($static, $dynamic);
}

/**
 * Select proxy for a monitor:
 * 1) If monitor has proxy_id and that proxy is enabled -> use it
 * 2) Else: find first enabled proxy with same location
 * 3) Else: no proxy (null)
 */
function select_proxy_for_monitor(array $monitor): ?array
{
    $location = $monitor['location'] ?? null;
    $proxyId = $monitor['proxy_id'] ?? null;

    $all = get_all_proxies();

    if ($proxyId) {
        foreach ($all as $p) {
            if (!empty($p['enabled']) && $p['id'] === $proxyId) {
                return $p;
            }
        }
    }

    if ($location) {
        foreach ($all as $p) {
            if (!empty($p['enabled']) && strcasecmp($p['location'] ?? '', $location) === 0) {
                return $p;
            }
        }
    }

    return null;
}

/**
 * Build CURL handle with optional proxy
 */
function build_curl_handle(string $url, ?array $proxy, bool $needBody = false)
{
    $ch = curl_init($url);

    $opts = [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_FOLLOWLOCATION => true,
        CURLOPT_NOBODY => $needBody ? false : true,
        CURLOPT_TIMEOUT => 15,
        CURLOPT_CONNECTTIMEOUT => 5,
        CURLOPT_SSL_VERIFYPEER => false,
        CURLOPT_SSL_VERIFYHOST => 0,
    ];

    if ($proxy && !empty($proxy['proxy_url'])) {
        $proxyUrl = $proxy['proxy_url'];
        $parts = parse_url($proxyUrl);

        if (!empty($parts['host']) && !empty($parts['port'])) {
            $proxyHost = $parts['host'] . ':' . $parts['port'];
            $opts[CURLOPT_PROXY] = $proxyHost;

            if (!empty($parts['user'])) {
                $user = $parts['user'];
                $pass = $parts['pass'] ?? '';
                $opts[CURLOPT_PROXYUSERPWD] = $user . ':' . $pass;
            }

            if (!empty($parts['scheme']) && strtolower($parts['scheme']) === 'https') {
                $opts[CURLOPT_PROXYTYPE] = CURLPROXY_HTTP;
            }
        } else {
            // direct style http://user:pass@host:port - pass string directly
            $opts[CURLOPT_PROXY] = $proxyUrl;
        }
    }

    curl_setopt_array($ch, $opts);

    return $ch;
}

/**
 * ---------------------------
 * Formatting & history
 * ---------------------------
 */

function format_timestamp(?int $ts): string
{
    if (!$ts) return '-';
    return date('Y-m-d H:i:s', $ts);
}

function record_check_result(array $monitor, string $status, ?float $responseTime = null, ?string $message = null): array
{
    $monitor['last_check'] = now();
    $monitor['last_status'] = $status;
    $monitor['last_message'] = $message;
    $monitor['last_response_time'] = $responseTime;

    if (!isset($monitor['history']) || !is_array($monitor['history'])) {
        $monitor['history'] = [];
    }

    $monitor['history'][] = [
        'time' => $monitor['last_check'],
        'status' => $status,
        'response_time' => $responseTime,
    ];

    if (count($monitor['history']) > 80) {
        $monitor['history'] = array_slice($monitor['history'], -80);
    }

    return $monitor;
}

function calculate_uptime_percentage(array $monitor): float
{
    if (empty($monitor['history'])) return 100.0;
    $total = count($monitor['history']);
    $up = 0;
    foreach ($monitor['history'] as $entry) {
        if ($entry['status'] === 'up') $up++;
    }
    return $total ? round(($up / $total) * 100, 2) : 100.0;
}

/**
 * ---------------------------
 * Check implementations
 * ---------------------------
 */

/**
 * HTTP / Endpoint monitoring + Keyword + Response time
 * Multi-location: uses proxy (if selected) to simulate location.
 */
function check_http(array $monitor): array
{
    $url = $monitor['target'];
    $keyword = $monitor['options']['keyword'] ?? null;

    $proxy = select_proxy_for_monitor($monitor);
    $needBody = $keyword !== null && $keyword !== '';

    $ch = build_curl_handle($url, $proxy, $needBody);
    $start = microtime(true);
    $body = curl_exec($ch);
    $info = curl_getinfo($ch);
    $err = curl_error($ch);
    curl_close($ch);
    $time = (microtime(true) - $start) * 1000;

    if ($err) {
        return ['status' => 'down', 'response_time' => $time, 'message' => $err];
    }

    $code = $info['http_code'] ?? 0;
    if ($code < 200 || $code >= 400) {
        return ['status' => 'down', 'response_time' => $time, 'message' => "HTTP $code"];
    }

    if ($keyword) {
        if (strpos((string)$body, $keyword) === false) {
            return [
                'status' => 'down',
                'response_time' => $time,
                'message' => "Keyword '{$keyword}' not found",
            ];
        }
    }

    return ['status' => 'up', 'response_time' => $time, 'message' => "OK (HTTP $code)"];
}

/**
 * Ping monitoring (from this server location)
 */
function check_ping(array $monitor): array
{
    $host = $monitor['target'];
    $cmd = sprintf('ping -c 1 -W 1 %s 2>&1', escapeshellarg($host));
    $start = microtime(true);
    @exec($cmd, $output, $exitCode);
    $time = (microtime(true) - $start) * 1000;

    if ($exitCode !== 0) {
        return ['status' => 'down', 'response_time' => $time, 'message' => "Ping failed"];
    }

    return ['status' => 'up', 'response_time' => $time, 'message' => "Ping OK"];
}

/**
 * Port monitoring (TCP)
 */
function check_port(array $monitor): array
{
    $host = $monitor['options']['host'] ?? null;
    $port = (int)($monitor['options']['port'] ?? 0);

    if (!$host || !$port) {
        return ['status' => 'down', 'response_time' => null, 'message' => "Host/port not set"];
    }

    $start = microtime(true);
    $errno = $errstr = null;
    $conn = @fsockopen($host, $port, $errno, $errstr, 5);
    $time = (microtime(true) - $start) * 1000;

    if (!$conn) {
        return ['status' => 'down', 'response_time' => $time, 'message' => "Port closed or unreachable ($errstr)"];
    }

    fclose($conn);
    return ['status' => 'up', 'response_time' => $time, 'message' => "Port open"];
}

/**
 * Cron job monitoring (heartbeat)
 */
function check_cron(array $monitor): array
{
    $interval = (int)($monitor['options']['interval_seconds'] ?? 600);
    $last = $monitor['options']['last_heartbeat'] ?? null;

    if (!$last) {
        return ['status' => 'down', 'response_time' => null, 'message' => "No heartbeat received yet"];
    }

    $diff = now() - $last;

    if ($diff > $interval) {
        return ['status' => 'down', 'response_time' => null, 'message' => "No heartbeat in last {$interval}s"];
    }

    return ['status' => 'up', 'response_time' => null, 'message' => "Heartbeat OK ({$diff}s ago)"];
}

/**
 * Domain monitoring (basic DNS A/AAAA existence)
 */
function check_domain(array $monitor): array
{
    $domain = $monitor['target'];
    $hasA = checkdnsrr($domain, 'A');
    $hasAAAA = checkdnsrr($domain, 'AAAA');

    if (!$hasA && !$hasAAAA) {
        return ['status' => 'down', 'response_time' => null, 'message' => "No A/AAAA DNS records"];
    }

    return ['status' => 'up', 'response_time' => null, 'message' => "DNS records OK"];
}

/**
 * DNS monitoring for specific record type (A/AAAA/CNAME/MX...)
 */
function check_dns(array $monitor): array
{
    $domain = $monitor['target'];
    $type = strtoupper($monitor['options']['record_type'] ?? 'A');

    $has = checkdnsrr($domain, $type);

    if (!$has) {
        return ['status' => 'down', 'response_time' => null, 'message' => "No {$type} DNS record"];
    }

    return ['status' => 'up', 'response_time' => null, 'message' => "{$type} record OK"];
}

/**
 * SSL monitoring (expiry days) with optional proxy (multi-location)
 */
function check_ssl(array $monitor): array
{
    $host = $monitor['target'];
    $port = 443;

    // For SSL check we still connect directly from server;
    // if you want to route via HTTP CONNECT proxy, you'd need more complex setup.
    // Here we keep it simple and local.
    $start = microtime(true);
    $context = stream_context_create([
        'ssl' => [
            'capture_peer_cert' => true,
            'verify_peer' => false,
            'verify_peer_name' => false,
        ]
    ]);

    $client = @stream_socket_client(
        "ssl://{$host}:{$port}",
        $errno,
        $errstr,
        10,
        STREAM_CLIENT_CONNECT,
        $context
    );
    $time = (microtime(true) - $start) * 1000;

    if (!$client) {
        return ['status' => 'down', 'response_time' => $time, 'message' => "SSL connection failed: $errstr"];
    }

    $params = stream_context_get_params($client);
    fclose($client);

    if (empty($params['options']['ssl']['peer_certificate'])) {
        return ['status' => 'down', 'response_time' => $time, 'message' => "No certificate found"];
    }

    $cert = openssl_x509_parse($params['options']['ssl']['peer_certificate']);
    $validTo = $cert['validTo_time_t'] ?? null;
    if (!$validTo) {
        return ['status' => 'down', 'response_time' => $time, 'message' => "Cannot read expiry"];
    }

    $daysLeft = floor(($validTo - now()) / 86400);

    if ($daysLeft < 0) {
        return ['status' => 'down', 'response_time' => $time, 'message' => "Certificate expired " . abs($daysLeft) . " days ago"];
    }

    if ($daysLeft < 7) {
        return ['status' => 'down', 'response_time' => $time, 'message' => "Certificate expiring in {$daysLeft} days"];
    }

    return ['status' => 'up', 'response_time' => $time, 'message' => "SSL OK ({$daysLeft} days left)"];
}
