<?php
// admin.php - simple admin panel for managing monitors + proxies

require __DIR__ . '/functions.php';
global $config;

session_start();

// Simple password protection
if (empty($_SESSION['admin_authenticated'])) {
    if ($_SERVER['REQUEST_METHOD'] === 'POST') {
        $pass = $_POST['password'] ?? '';
        if ($pass === $config['admin_password']) {
            $_SESSION['admin_authenticated'] = true;
            header('Location: admin.php');
            exit;
        } else {
            $error = "Wrong password";
        }
    }

    ?>
    <!doctype html>
    <html lang="en">
    <head>
        <meta charset="utf-8">
        <title>Admin Login</title>
        <link rel="stylesheet" href="assets/style.css">
    </head>
    <body class="auth-page">
    <div class="auth-box">
        <h1>Admin</h1>
        <?php if (!empty($error)): ?>
            <div class="alert alert-error"><?= htmlspecialchars($error) ?></div>
        <?php endif; ?>
        <form method="post">
            <label>Password</label>
            <input type="password" name="password" required>
            <button type="submit">Login</button>
        </form>
    </div>
    </body>
    </html>
    <?php
    exit;
}

// Authenticated
$monitors = load_monitors();
$dynamicProxies = load_dynamic_proxies();
$allProxies = get_all_proxies();

// Actions
$action = $_GET['action'] ?? 'list';

/**
 * Save monitor
 */
if ($action === 'save' && $_SERVER['REQUEST_METHOD'] === 'POST') {
    $id = $_POST['id'] ?: generate_id();

    $monitor = [
        'id' => $id,
        'name' => trim($_POST['name'] ?? ''),
        'type' => $_POST['type'] ?? 'http',
        'target' => trim($_POST['target'] ?? ''),
        'location' => trim($_POST['location'] ?? ''),
        'paused' => !empty($_POST['paused']),
        'proxy_id' => $_POST['proxy_id'] ?? '',
        'options' => [],
        'history' => [],
        'last_check' => null,
        'last_status' => null,
        'last_message' => null,
        'last_response_time' => null,
    ];

    switch ($monitor['type']) {
        case 'keyword':
            $monitor['options']['keyword'] = trim($_POST['keyword'] ?? '');
            break;
        case 'port':
            $monitor['options']['host'] = trim($_POST['host'] ?? '');
            $monitor['options']['port'] = (int)($_POST['port'] ?? 0);
            break;
        case 'cron':
            $monitor['options']['interval_seconds'] = (int)($_POST['interval_seconds'] ?? 600);
            break;
        case 'dns':
            $monitor['options']['record_type'] = trim($_POST['record_type'] ?? 'A');
            break;
    }

    update_monitor_in_list($monitors, $monitor);
    save_monitors($monitors);

    header('Location: admin.php');
    exit;
}

/**
 * Delete monitor
 */
if ($action === 'delete') {
    $id = $_GET['id'] ?? '';
    $monitors = array_values(array_filter($monitors, fn($m) => $m['id'] !== $id));
    save_monitors($monitors);
    header('Location: admin.php');
    exit;
}

// Determine monitor to edit
$editMonitor = null;
if ($action === 'edit') {
    $id = $_GET['id'] ?? '';
    $editMonitor = find_monitor($monitors, $id);
}

/**
 * Save dynamic proxy (B3)
 */
if ($action === 'proxy_save' && $_SERVER['REQUEST_METHOD'] === 'POST') {
    $name = trim($_POST['proxy_name'] ?? '');
    $location = trim($_POST['proxy_location'] ?? '');
    $url = trim($_POST['proxy_url'] ?? '');
    $enabled = !empty($_POST['proxy_enabled']);

    if ($name && $location && $url) {
        $dynamicProxies[] = [
            'id' => generate_id(),
            'name' => $name,
            'location' => $location,
            'proxy_url' => $url,
            'enabled' => $enabled,
            'source' => 'dynamic',
        ];
        save_dynamic_proxies($dynamicProxies);
    }

    header('Location: admin.php');
    exit;
}

/**
 * Delete dynamic proxy
 */
if ($action === 'proxy_delete') {
    $id = $_GET['id'] ?? '';
    $dynamicProxies = array_values(array_filter($dynamicProxies, fn($p) => $p['id'] !== $id));
    save_dynamic_proxies($dynamicProxies);
    header('Location: admin.php');
    exit;
}

?>
<!doctype html>
<html lang="en">
<head>
    <meta charset="utf-8">
    <title>Uptime Admin</title>
    <link rel="stylesheet" href="assets/style.css">
</head>
<body class="admin">
<header class="topbar">
    <div class="brand"><?= htmlspecialchars($config['brand_name']) ?> - Admin</div>
    <nav>
        <a href="status.php" target="_blank">View Status Page</a>
        <a href="admin.php">Monitors</a>
    </nav>
</header>

<main class="container">
    <section class="card">
        <h2><?= $editMonitor ? 'Edit Monitor' : 'Add Monitor' ?></h2>
        <form method="post" action="admin.php?action=save" class="grid-form">
            <input type="hidden" name="id" value="<?= htmlspecialchars($editMonitor['id'] ?? '') ?>">

            <div>
                <label>Name</label>
                <input type="text" name="name" required value="<?= htmlspecialchars($editMonitor['name'] ?? '') ?>">
            </div>

            <div>
                <label>Type</label>
                <select name="type" required>
                    <?php
                    $types = [
                        'http' => 'Website / Endpoint',
                        'keyword' => 'Keyword (HTTP)',
                        'ping' => 'Ping',
                        'port' => 'Port',
                        'cron' => 'Cron Job',
                        'domain' => 'Domain',
                        'dns' => 'DNS Record',
                        'ssl' => 'SSL Certificate',
                    ];
                    $currentType = $editMonitor['type'] ?? 'http';
                    foreach ($types as $key => $label): ?>
                        <option value="<?= $key ?>" <?= $currentType === $key ? 'selected' : '' ?>>
                            <?= htmlspecialchars($label) ?>
                        </option>
                    <?php endforeach; ?>
                </select>
            </div>

            <div>
                <label>Target (URL / Host / Domain)</label>
                <input type="text" name="target" required value="<?= htmlspecialchars($editMonitor['target'] ?? '') ?>">
            </div>

            <div>
                <label>Location (e.g. EU, US, ASIA)</label>
                <input type="text" name="location" value="<?= htmlspecialchars($editMonitor['location'] ?? '') ?>">
            </div>

            <div>
                <label>Proxy</label>
                <select name="proxy_id">
                    <option value="">Auto (by location)</option>
                    <?php
                    $selectedProxyId = $editMonitor['proxy_id'] ?? '';
                    foreach ($allProxies as $p):
                        $id = $p['id'] ?? '';
                        $label = ($p['location'] ?? '') . ' - ' . ($p['name'] ?? '');
                        ?>
                        <option value="<?= htmlspecialchars($id) ?>" <?= $selectedProxyId === $id ? 'selected' : '' ?>>
                            <?= htmlspecialchars($label) ?>
                        </option>
                    <?php endforeach; ?>
                </select>
            </div>

            <div>
                <label>
                    <input type="checkbox" name="paused" <?= !empty($editMonitor['paused']) ? 'checked' : '' ?>>
                    Paused
                </label>
            </div>

            <div class="full-width options-info">
                <strong>Type-specific options:</strong>
                <ul>
                    <li><b>Keyword</b>: uses "Keyword" field below.</li>
                    <li><b>Port</b>: uses Host + Port fields.</li>
                    <li><b>Cron</b>: uses Interval seconds (for heartbeat check).</li>
                    <li><b>DNS</b>: uses Record Type (A, AAAA, CNAME, MX...).</li>
                </ul>
            </div>

            <div>
                <label>Keyword (for keyword type)</label>
                <input type="text" name="keyword" value="<?= htmlspecialchars($editMonitor['options']['keyword'] ?? '') ?>">
            </div>

            <div>
                <label>Host (for port type)</label>
                <input type="text" name="host" value="<?= htmlspecialchars($editMonitor['options']['host'] ?? '') ?>">
            </div>

            <div>
                <label>Port (for port type)</label>
                <input type="number" name="port" value="<?= htmlspecialchars($editMonitor['options']['port'] ?? '') ?>">
            </div>

            <div>
                <label>Cron Interval (seconds)</label>
                <input type="number" name="interval_seconds" value="<?= htmlspecialchars($editMonitor['options']['interval_seconds'] ?? 600) ?>">
            </div>

            <div>
                <label>DNS Record Type</label>
                <input type="text" name="record_type" placeholder="A / AAAA / CNAME / MX..." value="<?= htmlspecialchars($editMonitor['options']['record_type'] ?? 'A') ?>">
            </div>

            <div class="full-width">
                <button type="submit">Save Monitor</button>
            </div>
        </form>
    </section>

    <section class="card">
        <h2>Monitors</h2>
        <table class="table">
            <thead>
            <tr>
                <th>Name</th>
                <th>Type</th>
                <th>Target</th>
                <th>Location</th>
                <th>Proxy</th>
                <th>Status</th>
                <th>Last Check</th>
                <th>Uptime</th>
                <th></th>
            </tr>
            </thead>
            <tbody>
            <?php foreach ($monitors as $m):
                $uptime = calculate_uptime_percentage($m);
                $status = $m['last_status'] ?? 'unknown';
                $badgeClass = $status === 'up' ? 'badge-up' : ($status === 'down' ? 'badge-down' : 'badge-unknown');
                $proxyLabel = '';
                if (!empty($m['proxy_id'])) {
                    foreach ($allProxies as $p) {
                        if (($p['id'] ?? '') === $m['proxy_id']) {
                            $proxyLabel = ($p['location'] ?? '') . ' - ' . ($p['name'] ?? '');
                            break;
                        }
                    }
                }
                ?>
                <tr>
                    <td><?= htmlspecialchars($m['name']) ?></td>
                    <td><?= htmlspecialchars($m['type']) ?></td>
                    <td><?= htmlspecialchars($m['target']) ?></td>
                    <td><?= htmlspecialchars($m['location'] ?? '-') ?></td>
                    <td><?= $proxyLabel ? htmlspecialchars($proxyLabel) : 'Auto' ?></td>
                    <td>
                        <span class="badge <?= $badgeClass ?>">
                            <?= strtoupper($status) ?>
                        </span>
                    </td>
                    <td><?= format_timestamp($m['last_check'] ?? null) ?></td>
                    <td><?= $uptime ?>%</td>
                    <td>
                        <a href="admin.php?action=edit&id=<?= urlencode($m['id']) ?>">Edit</a> ·
                        <a href="admin.php?action=delete&id=<?= urlencode($m['id']) ?>" onclick="return confirm('Delete monitor?')">Delete</a>
                    </td>
                </tr>
            <?php endforeach; ?>
            <?php if (empty($monitors)): ?>
                <tr><td colspan="9">No monitors yet.</td></tr>
            <?php endif; ?>
            </tbody>
        </table>
    </section>

    <section class="card">
        <h2>Dynamic Proxies (B3)</h2>
        <p style="font-size: 0.8rem; color: #94a3b8;">
            Here you can define proxies per location. Monitors will automatically pick a proxy matching their location,
            unless a specific proxy is selected.
        </p>
        <form method="post" action="admin.php?action=proxy_save" class="grid-form">
            <div>
                <label>Name</label>
                <input type="text" name="proxy_name" required placeholder="EU Proxy #1">
            </div>
            <div>
                <label>Location</label>
                <input type="text" name="proxy_location" required placeholder="EU / US / ASIA">
            </div>
            <div>
                <label>Proxy URL</label>
                <input type="text" name="proxy_url" required placeholder="http://user:pass@host:port">
            </div>
            <div>
                <label>
                    <input type="checkbox" name="proxy_enabled" checked>
                    Enabled
                </label>
            </div>
            <div class="full-width">
                <button type="submit">Add Proxy</button>
            </div>
        </form>

        <table class="table" style="margin-top: 12px;">
            <thead>
            <tr>
                <th>Name</th>
                <th>Location</th>
                <th>URL</th>
                <th>Enabled</th>
                <th>Source</th>
                <th></th>
            </tr>
            </thead>
            <tbody>
            <?php foreach (get_all_proxies() as $p): ?>
                <tr>
                    <td><?= htmlspecialchars($p['name'] ?? '') ?></td>
                    <td><?= htmlspecialchars($p['location'] ?? '') ?></td>
                    <td><code style="font-size:0.7rem;"><?= htmlspecialchars($p['proxy_url'] ?? '') ?></code></td>
                    <td><?= !empty($p['enabled']) ? 'Yes' : 'No' ?></td>
                    <td><?= htmlspecialchars($p['source'] ?? 'static') ?></td>
                    <td>
                        <?php if (($p['source'] ?? 'static') === 'dynamic'): ?>
                            <a href="admin.php?action=proxy_delete&id=<?= urlencode($p['id']) ?>" onclick="return confirm('Delete proxy?')">Delete</a>
                        <?php else: ?>
                            -
                        <?php endif; ?>
                    </td>
                </tr>
            <?php endforeach; ?>
            <?php if (empty(get_all_proxies())): ?>
                <tr><td colspan="6">No proxies defined yet. Add some above or static ones in config.php.</td></tr>
            <?php endif; ?>
            </tbody>
        </table>
    </section>
</main>
</body>
</html>
